#pragma semicolon 1
#pragma newdecls required

#include <sourcemod>
#include <sdkhooks>
#include <sdktools_trace>
#include <sdktools_functions>
#include <tf2_stocks>
#include <clientprefs>

#define PLUGIN_AUTHOR  "ack"
#define PLUGIN_VERSION "0.2"

#define CONFIG_FILE     "configs/eotl_og.cfg"

public Plugin myinfo = {
	name = "eotl_og",
	author = PLUGIN_AUTHOR,
	description = "Granny Sound Bites",
	version = PLUGIN_VERSION,
	url = ""
};

enum struct PlayerState {
    bool enabled;
    int playCount;
}

StringMap g_smSoundBites;
StringMap g_smDescriptions;
ArrayList g_alShortNames;
float g_fLastPlay;
ConVar g_cvMaxPlayerPlays;
ConVar g_cvMinTime;
Handle g_hClientCookies;
PlayerState g_PlayerStates[MAXPLAYERS + 1];


public void OnPluginStart() {
    LogMessage("version %s starting", PLUGIN_VERSION);
    RegConsoleCmd("sm_og", CommandOG);

    g_cvMaxPlayerPlays = CreateConVar("eotl_og_max_player_plays", "2", "maximum number of times a player can play a og per map");
    g_cvMinTime  = CreateConVar("eotl_og_min_time", "10.0", "number of seconds that must pass before another og can be played");

    g_alShortNames = CreateArray(16);
    g_hClientCookies = RegClientCookie("og enabled", "og enabled", CookieAccess_Private);
}

public void OnMapStart() {
    g_smSoundBites = CreateTrie();
    g_smDescriptions = CreateTrie();
    g_alShortNames.Clear();

    g_fLastPlay = 0.0;
    for(int client = 1; client <= MaxClients; client++) {
        g_PlayerStates[client].playCount = 0;
        g_PlayerStates[client].enabled = false;
    }

    LoadConfig();
}

public void OnMapEnd() {
    CloseHandle(g_smSoundBites);
    CloseHandle(g_smDescriptions);
}

public void OnClientConnected(int client) {
    g_PlayerStates[client].playCount = 0;
    g_PlayerStates[client].enabled = false;
}

public void OnClientCookiesCached(int client) {
   LoadClientConfig(client);
}

public void OnClientPostAdminCheck(int client) {
    LoadClientConfig(client);
}

public Action CommandOG(int client, int args) {
    char shortName[16];
    char soundFile[PLATFORM_MAX_PATH];

    if(args > 1) {
        PrintToChat(client, "\x01[\x03og\x01] Invalid syntax");
        return Plugin_Handled;
    }

    if(args == 0) {
        // pick random shortname
        // doing this in one lines seems to be less random?
        int rand = GetURandomInt();
        int index = rand % g_alShortNames.Length;
        g_alShortNames.GetString(index, shortName, sizeof(shortName));        
    } else {
        GetCmdArg(1, shortName, sizeof(shortName));
        StringToLower(shortName);
    }

    // disable sounds for client if they !og disable
    if(StrEqual(shortName, "disable")) {
        if(!g_PlayerStates[client].enabled) {
            PrintToChat(client, "\x01[\x03og\x01] is already \x03disabled\x01 for you");
            return Plugin_Handled;
        }
        g_PlayerStates[client].enabled = false;
        SaveClientConfig(client);
        PrintToChat(client, "\x01[\x03og\x01] sounds are now \x03disabled\x01 for you");
        return Plugin_Handled;
    }

    // auto-enable if user ran any other !og based command
    if(!g_PlayerStates[client].enabled) {
        g_PlayerStates[client].enabled = true;
        SaveClientConfig(client);
        PrintToChat(client, "\x01[\x03og\x01] is now \x03enabled\x01 for you");
    }

    if(StrEqual(shortName, "list")) {
        SendSoundBiteList(client);
        return Plugin_Handled;
    }

    if(!g_smSoundBites.GetString(shortName, soundFile, sizeof(soundFile))) {
        PrintToChat(client, "\x01[\x03og\x01] error invalid soundbite %s, use \"!og list\" for a list or \"!og disable\" to disable sounds", shortName);
        return Plugin_Handled;
    }

    // see if the user is allowed to play a og
    if(g_PlayerStates[client].playCount >= g_cvMaxPlayerPlays.IntValue) {
        PrintToChat(client,"\x01[\x03og\x01] sorry you are limited to %d og's per map", g_cvMaxPlayerPlays.IntValue);
        return Plugin_Handled;
    }

    float timeDiff = GetGameTime() - g_fLastPlay;
    if(timeDiff < g_cvMinTime.FloatValue) {
        PrintToChat(client, "\x01[\x03og\x01] You must wait %.1f seconds before another og will be allowed", g_cvMinTime.FloatValue - timeDiff);
        return Plugin_Handled;
    }

    g_PlayerStates[client].playCount++;
    g_fLastPlay = GetGameTime();

    char description[32];
    g_smDescriptions.GetString(shortName, description, sizeof(description));
    PrintToChat(client, "\x01[\x03og\x01] Playing %s", description);
    LogMessage("Playing shortname: %s, file: %s, description: %s", shortName, soundFile, description);
    PlaySound(soundFile);
    return Plugin_Handled;
}

// play the sound for the clients that have og enabled
void PlaySound(const char [] soundFile) {
    int client;

    for(client = 1; client <= MaxClients; client++) {

        if(!IsClientInGame(client)) {
            continue;
        }

        if(IsFakeClient(client)) {
            continue;
        }

        if(!g_PlayerStates[client].enabled) {
            continue;
        }

        EmitSoundToClient(client, soundFile);
    }
}

void SendSoundBiteList(int client) {
    char shortName[16];
    char description[32];

    PrintToChat(client, "\x03GrannyMcFannyBananyWHAMMY! Sound Bites\x01:");
    PrintToChat(client, "\x03  !og\x01    - random sound bite");
    for(int i = 0; i < g_alShortNames.Length;i++) {
        g_alShortNames.GetString(i, shortName, sizeof(shortName));
        g_smDescriptions.GetString(shortName, description, sizeof(description));
        PrintToChat(client, "\x03  !og %s\x01 - %s", shortName, description);
    }
}

void LoadConfig() {
    KeyValues cfg = CreateKeyValues("og");

    char configFile[PLATFORM_MAX_PATH];
    BuildPath(Path_SM, configFile, sizeof(configFile), CONFIG_FILE);

    LogMessage("loading config file: %s", configFile);
    if(!FileToKeyValues(cfg, configFile)) {
        SetFailState("unable to load config file!");
        return;
    }

    char shortName[16];
    char description[32];
    char soundFile[PLATFORM_MAX_PATH];
    char downloadFile[PLATFORM_MAX_PATH];
    if(cfg.JumpToKey("soundBites")) {
        KvGotoFirstSubKey(cfg);
        do {

            cfg.GetSectionName(shortName, sizeof(shortName));
            cfg.GetString("description", description, sizeof(description));
            cfg.GetString("soundBite", soundFile, sizeof(soundFile));
            g_smSoundBites.SetString(shortName, soundFile);
            g_smDescriptions.SetString(shortName, description);
            g_alShortNames.PushString(shortName);

            Format(downloadFile, sizeof(downloadFile), "sound/%s", soundFile);
            AddFileToDownloadsTable(downloadFile);
            PrecacheSound(soundFile, true);
            LogMessage("loaded %s = %s with description %s", shortName, soundFile, description);
        } while(KvGotoNextKey(cfg));
    }
    CloseHandle(cfg);  
}

void LoadClientConfig(int client) {

    if(IsFakeClient(client)) {
        return;
    }

    if(!IsClientInGame(client)) {
        return;
	}

    char enableState[6];
    GetClientCookie(client, g_hClientCookies, enableState, 6);
    if(StrEqual(enableState, "true")) {
        g_PlayerStates[client].enabled = true;
    } else {
        g_PlayerStates[client].enabled = false;
    }

    LogMessage("client: %d og enable state %d", client, g_PlayerStates[client].enabled);
}

void SaveClientConfig(int client) {
    char enableState[6];
    if(g_PlayerStates[client].enabled) {
        Format(enableState, 6, "true");
    } else {
        Format(enableState, 6, "false");
    }

    LogMessage("client: %d, saving og enable state as %s", client, enableState);
    SetClientCookie(client, g_hClientCookies, enableState);
}

void StringToLower(char[] string) {
    int len = strlen(string);
    int i;

    for(i = 0;i < len;i++) {
        string[i] = CharToLower(string[i]);
    }
}