#pragma semicolon 1
#pragma newdecls required

#define DEBUG

#define PLUGIN_AUTHOR "ack"
#define PLUGIN_VERSION "0.04"

#include <sourcemod>
#include <tf2>
#include <tf2_stocks>
#include <sdktools>
#include <sdkhooks>

public Plugin myinfo = {
	name = "Eotl Respawn",
	author = PLUGIN_AUTHOR,
	description = "Eotl Respawn",
	version = PLUGIN_VERSION,
	url = ""
};

enum struct PlayerState {
	float spawnTimeLast;
	float spawnTimeNext;
	float penalty;
}
PlayerState ga_sPlayerStates[MAXPLAYERS + 1];

bool g_bSpawnEnabled = false;
int g_iTF2GameRulesEntity;
int g_iPlayerResourceEntity;

ConVar g_cvEnabled;
ConVar g_cvBlueSettingRespawnMin;
ConVar g_cvBlueSettingPenaltyMax;
ConVar g_cvBlueSettingPenaltyAmount;
ConVar g_cvBlueSettingAliveThreshold;
ConVar g_cvRedSettingRespawnMin;
ConVar g_cvRedSettingPenaltyMax;
ConVar g_cvRedSettingPenaltyAmount;
ConVar g_cvRedSettingAliveThreshold;

public void OnPluginStart() {
	PrintToServer("Eotl Respawn: Version %s Loaded", PLUGIN_VERSION);

	HookEvent("player_death", EventPlayerDeath);
	HookEvent("player_spawn", EventPlayerSpawn);
	HookEvent("teamplay_round_start", EventRoundStart, EventHookMode_PostNoCopy);

	HookEvent("teamplay_round_stalemate", EventDisableSpawn, EventHookMode_PostNoCopy);
	HookEvent("teamplay_round_win", EventDisableSpawn, EventHookMode_PostNoCopy);
	HookEvent("teamplay_game_over", EventDisableSpawn, EventHookMode_PostNoCopy);
	
	g_cvEnabled = CreateConVar("sm_respawn_time_enabled", "1", "Enable or disable the plugin 1=On, 0=Off", FCVAR_NOTIFY, true, 0.0, true, 1.0);
	
	g_cvBlueSettingRespawnMin = CreateConVar("sm_respawn_time_blue", "10", "blue teams minimum spawn time", FCVAR_NOTIFY, true, 0.0);
	g_cvBlueSettingPenaltyMax = CreateConVar("sm_eotl_respawn_blue_penalty_max", "10", "blue team maximum penalty time", FCVAR_NOTIFY, true, 0.0);
	g_cvBlueSettingPenaltyAmount = CreateConVar("sm_eotl_respawn_blue_penalty_amount", "2", "blue teams penalty amount", FCVAR_NOTIFY, true, 0.0);
	g_cvBlueSettingAliveThreshold = CreateConVar("sm_eotl_respawn_blue_life_threshold", "20", "blue teams alive threshold", FCVAR_NOTIFY, true, 1.0);

	g_cvRedSettingRespawnMin = CreateConVar("sm_respawn_time_red", "10", "red teams minimum spawn time", FCVAR_NOTIFY, true, 0.0);
	g_cvRedSettingPenaltyMax = CreateConVar("sm_eotl_respawn_red_penalty_max", "10", "red team maximum penalty time", FCVAR_NOTIFY, true, 0.0);
	g_cvRedSettingPenaltyAmount = CreateConVar("sm_eotl_respawn_red_penalty_amount", "2", "red teams penalty amount", FCVAR_NOTIFY, true, 0.0);
	g_cvRedSettingAliveThreshold = CreateConVar("sm_eotl_respawn_red_life_threshold", "20", "red teams alive threshold", FCVAR_NOTIFY, true, 1.0);
}

public void OnMapStart() {
	
	//PrintToServer("Eotl Repawn (%.1f): OnMapStart, resetting all player penalties", GetGameTime());

	// force disable if arena mode
	if(FindEntityByClassname(-1, "tf_logic_arena") >= 0) {
		SetConVarInt(g_cvEnabled, 0);
	}
	
	g_iTF2GameRulesEntity = FindEntityByClassname(-1, "tf_gamerules");
	if(g_iTF2GameRulesEntity == -1) {
		//PrintToServer("Eotl Repawn (%.1f): Couldn't find tf_gamerules (won't be able to disable spawn waves)", GetGameTime());
	}
	
	g_iPlayerResourceEntity = GetPlayerResourceEntity();
	if(g_iPlayerResourceEntity == -1) {
		//PrintToServer("Eotl Respawn (%.1f): Couldn't get PlayerResourceEntity (won't be able to override respawn value in client huds", GetGameTime());
	}
	
	g_bSpawnEnabled = true;
}

public Action EventRoundStart(Handle event, const char[] name, bool dontBroadcast) {
	g_bSpawnEnabled = true;
	float gameTime = GetGameTime();

	//PrintToServer("Eotl Respawn (%.1f): round start, resetting all player penalties", gameTime);	
	for (int client = 1; client <= MaxClients; client++) {
		ga_sPlayerStates[client].spawnTimeLast = gameTime;
		ga_sPlayerStates[client].spawnTimeNext = 0.0;
		ga_sPlayerStates[client].penalty = 0.0;
	}
	
	return Plugin_Continue;
}

// events that should trigger us to stop (re)spawning players
public Action EventDisableSpawn(Handle event, const char[] name, bool dontBroadcast) {
	g_bSpawnEnabled = false;
}

public Action EventPlayerSpawn(Handle event, const char[] name, bool dontBroadcast) {
	int client = GetClientOfUserId(GetEventInt(event, "userid"));
	TFTeam team = TF2_GetClientTeam(client);
	
	if (team == TFTeam_Blue || team == TFTeam_Red) {
		ga_sPlayerStates[client].spawnTimeLast = GetGameTime();
		ga_sPlayerStates[client].spawnTimeNext = 0.0;
		//PrintToServer("Eotl Respawn (%.1f): client %d (team: %s) spawned", ga_sPlayerStates[client].spawnTimeLast, client, (team == TFTeam_Blue ? "blue" : "red"));
	}
	
	return Plugin_Continue;
}

public Action EventPlayerDeath(Handle event, const char[] name, bool dontBroadcast) {

	if(!g_cvEnabled.BoolValue || !g_bSpawnEnabled) {
		return Plugin_Continue;
	}
	
	if((GetEventInt(event, "death_flags") & TF_DEATHFLAG_DEADRINGER) == TF_DEATHFLAG_DEADRINGER) {
		return Plugin_Continue;
	}
	
	int client = GetClientOfUserId(GetEventInt(event, "userid"));
	TFTeam team = TF2_GetClientTeam(client);
	
	float respawnMin = 0.0;
	float penaltyMax = 0.0;
	float penaltyAmount = 0.0;
	float aliveThreshold = 0.0;
	
	if(team == TFTeam_Blue) {
		respawnMin = g_cvBlueSettingRespawnMin.FloatValue;
		penaltyMax = g_cvBlueSettingPenaltyMax.FloatValue;
		penaltyAmount = g_cvBlueSettingPenaltyAmount.FloatValue;
		aliveThreshold = g_cvBlueSettingAliveThreshold.FloatValue;
	} else if(team == TFTeam_Red) {
		respawnMin = g_cvRedSettingRespawnMin.FloatValue;
		penaltyMax = g_cvRedSettingPenaltyMax.FloatValue;
		penaltyAmount = g_cvRedSettingPenaltyAmount.FloatValue;
		aliveThreshold = g_cvRedSettingAliveThreshold.FloatValue;
	} else {
		return Plugin_Continue;
	}
	
	float respawnTimeLast = respawnMin + ga_sPlayerStates[client].penalty;
	float gameTime = GetGameTime();
	float aliveTime = gameTime - ga_sPlayerStates[client].spawnTimeLast;
	if(aliveTime < aliveThreshold) {
		ga_sPlayerStates[client].penalty = ga_sPlayerStates[client].penalty + penaltyAmount;
		if(ga_sPlayerStates[client].penalty > penaltyMax) {
			ga_sPlayerStates[client].penalty = penaltyMax;
		}
	} else {
		ga_sPlayerStates[client].penalty = ga_sPlayerStates[client].penalty - (penaltyAmount * (aliveTime / aliveThreshold));
		if(ga_sPlayerStates[client].penalty < 0.0) {
			ga_sPlayerStates[client].penalty = 0.0;
		}
	}
	
	float respawnLength = respawnMin + ga_sPlayerStates[client].penalty;
	ga_sPlayerStates[client].spawnTimeNext = gameTime + respawnLength + 1;
	
	SetSpawnWaves();
	
	//PrintToServer("Eotl Respawn (%.1f): client %d died (team: %s), life length was %.1fs (min: %.1fs), respawn in %.1fs (prev was: %.1fs)", gameTime, client, (team == TFTeam_Blue ? "blue" : "red"), aliveTime, aliveThreshold, respawnLength, respawnTimeLast);
	if(ga_sPlayerStates[client].penalty > 0.0) {
		PrintHintText(client, "Respawn penalty is %.1f seconds", ga_sPlayerStates[client].penalty);
	}
	
	SDKHook(client, SDKHook_SetTransmit, OverrideRespawnHud);
	CreateTimer(respawnLength, SpawnPlayerTimer, client, TIMER_FLAG_NO_MAPCHANGE);
	return Plugin_Continue;
}
	
public Action SpawnPlayerTimer(Handle timer, int client) {

	if(!g_bSpawnEnabled || !IsClientConnected(client) || !IsClientInGame(client) || IsPlayerAlive(client)) {
		return;
	}

	TFTeam team = TF2_GetClientTeam(client);
	if (team == TFTeam_Blue || team == TFTeam_Red) {
		//PrintToServer("Eotl Respawn (%.1f): spawning client %d (team: %s)", GetGameTime(), client, (team == TFTeam_Blue ? "blue" : "red"));
		TF2_RespawnPlayer(client);
	}
}

public void OnClientConnected(int client) {
	ga_sPlayerStates[client].spawnTimeLast = GetGameTime();
	ga_sPlayerStates[client].penalty = 0.0;
	//PrintToServer("Eotl Respawn (%.1f): client %d connected, init client's penalty", ga_sPlayerStates[client].spawnTimeLast, client);
}

// set spawn wave values to be min + max penalty or else players may spawn before
// we want them to
void SetSpawnWaves() {

	if(g_iTF2GameRulesEntity < 0) {
		return;
	}
	
	if(!g_cvEnabled.BoolValue) {
		return;
	}
	
	float respawnWave = g_cvBlueSettingRespawnMin.FloatValue + g_cvBlueSettingPenaltyMax.FloatValue;
	SetVariantFloat(respawnWave);
	AcceptEntityInput(g_iTF2GameRulesEntity, "SetBlueTeamRespawnWaveTime", -1, -1, 0);

	respawnWave = g_cvRedSettingRespawnMin.FloatValue + g_cvRedSettingPenaltyMax.FloatValue;
	SetVariantFloat(respawnWave);
	AcceptEntityInput(g_iTF2GameRulesEntity, "SetRedTeamRespawnWaveTime", -1, -1, 0);
}

// override the clients m_flNextRespawnTime so their hud shows the right info
public Action OverrideRespawnHud(int client, int other) {
	
	if(g_iPlayerResourceEntity == -1) {
		ga_sPlayerStates[client].spawnTimeNext = 0.0;
		SDKUnhook(client, SDKHook_SetTransmit, OverrideRespawnHud);
		return;
	}
	
	if(client == other) {
		// need to be aggressive about setting the value or it wont happen in time and have no effect
		SetEntPropFloat(g_iPlayerResourceEntity, Prop_Send, "m_flNextRespawnTime", ga_sPlayerStates[client].spawnTimeNext, client);
		
		if(!g_bSpawnEnabled || !IsClientConnected(client) || !IsClientInGame(client) || IsPlayerAlive(client)) {
			ga_sPlayerStates[client].spawnTimeNext = 0.0;
			SDKUnhook(client, SDKHook_SetTransmit, OverrideRespawnHud);
			return;
		}
	}
}