#pragma semicolon 1
#pragma newdecls required

#include <sourcemod>
#include <sdktools>
#include <tf2>

#define PLUGIN_AUTHOR  "ack"
#define PLUGIN_VERSION "0.4"

#define CONFIG_FILE    "configs/eotl_rolled.cfg"
#define KV_FLOAT_UNSET -3.14259
#define KV_INT_UNSET   -314259

public Plugin myinfo = {
	name = "eotl_rolled",
	author = PLUGIN_AUTHOR,
	description = "play a sound at the end of around if a team got rolled/stuffed",
	version = PLUGIN_VERSION,
	url = ""
};

ArrayList g_soundsRolled;
ArrayList g_soundsStuffed;
ConVar g_cvTimeRolled;
ConVar g_cvTimeStuffed;
ConVar g_cvCapStuffed;
ConVar g_cvDelay;

// per map configs provided in the config file
StringMap g_smTimeRolled;
StringMap g_smTimeStuffed;
StringMap g_smCapStuffed;

int g_caps;
float g_roundStartTime;

public void OnPluginStart() {
    LogMessage("version %s starting", PLUGIN_VERSION);

    g_cvTimeRolled = CreateConVar("eotl_rolled_time_rolled", "8", "If red team loses and this many minutes haven't passed, they are considered rolled");
    g_cvTimeStuffed = CreateConVar("eotl_rolled_time_stuffed", "-1", "If blue team loses and this many minutes haven't passed, they are considered stuffed");
    g_cvCapStuffed = CreateConVar("eotl_rolled_cap_stuffed", "1", "If blue team doen't cap this many points they are considered stuffed");
    g_cvDelay = CreateConVar("eotl_rolled_delay", "5.0", "Delay seconds before playing sound at end of round");

    g_soundsRolled = CreateArray(PLATFORM_MAX_PATH);
    g_soundsStuffed = CreateArray(PLATFORM_MAX_PATH);

    HookEvent("teamplay_round_start", EventRoundStart, EventHookMode_PostNoCopy);
    HookEvent("teamplay_round_win", EventRoundWin);
    HookEvent("teamplay_point_captured", EventPointCaptured, EventHookMode_PostNoCopy);

}


public void OnMapStart() {
    g_smTimeRolled = CreateTrie();
    g_smTimeStuffed = CreateTrie();
    g_smCapStuffed = CreateTrie();

    LoadConfig();

}

public void OnMapEnd() {
    CloseHandle(g_smTimeRolled);
    CloseHandle(g_smTimeStuffed);
    CloseHandle(g_smCapStuffed);
}

public Action EventRoundStart(Handle event, const char[] name, bool dontBroadcast) {
    g_roundStartTime = GetGameTime();
    g_caps = 0;
    return Plugin_Continue;
}

public Action EventRoundWin(Handle event, const char[] name, bool dontBroadcast) {
    float roundTime = (GetGameTime() - g_roundStartTime) / 60.0;
    TFTeam winTeam = view_as<TFTeam>(GetEventInt(event, "team"));
    
    float timeRolled = g_cvTimeRolled.FloatValue;
    float timeStuffed = g_cvTimeStuffed.FloatValue;
    int capStuffed = g_cvCapStuffed.IntValue;
    float delay = g_cvDelay.FloatValue;

    char mapName[PLATFORM_MAX_PATH];
    GetCurrentMap(mapName, PLATFORM_MAX_PATH);

    
    if(delay <= 0.0) {
        delay = 0.1;
    }

    // apply overrides from the config file for this map
    float overrideTime;
    int overrideCap;
    if(g_smTimeRolled.GetValue(mapName, overrideTime)) {
        timeRolled = overrideTime;
    }

    if(g_smTimeStuffed.GetValue(mapName, overrideTime)) {
        timeStuffed = overrideTime;
    }

    if(g_smCapStuffed.GetValue(mapName, overrideCap)) {
        capStuffed = overrideCap;
    }

    LogMessage("Map: %s, Round Time: %.2f minutes, Winning Team: %s, Caps: %d (Rolled Time: %.1f, Stuffed Time: %.1f, Stuffed Cap: %d)", mapName, roundTime, winTeam == TFTeam_Blue ? "blue" : "red", g_caps, timeRolled, timeStuffed, capStuffed);
   
    // blue team won, check for a roll
    if(winTeam == TFTeam_Blue) {
        if(timeRolled <= 0.0) {
            return Plugin_Continue;
        }

        if(roundTime > timeRolled) {
            return Plugin_Continue;
        }

        PrintToChatAll("\x01Red got \x03Rolled\x01!");
        
        if(g_soundsRolled.Length <= 0) {
            LogMessage("No Rolled sounds to play!");
            return Plugin_Continue;
        }

        // doing this in one lines seems to be less random?
        int rand = GetURandomInt();
        int index = rand % g_soundsRolled.Length;

        if(index < g_soundsRolled.Length) {
            CreateTimer(delay, PlaySoundRolled, index, TIMER_FLAG_NO_MAPCHANGE);
        }

    // red time won, check for stuffed
    } else if(winTeam == TFTeam_Red) {

        bool isStuffed = false;
        // time based
        if(timeStuffed > 0.0 && (roundTime < timeStuffed)) {
            isStuffed = true;
        }

        // didn't cap enough points
        if(capStuffed >= 0 && g_caps <= capStuffed) {
            isStuffed = true;
        }      

        if(!isStuffed) {
            return Plugin_Continue;
        }

        PrintToChatAll("\x01Blue got \x03Stuffed\x01!");

        if(g_soundsStuffed.Length <= 0) {
            LogMessage("No Stuffed sounds to play!");
            return Plugin_Continue;
        }

        int rand = GetURandomInt();
        int index = rand % g_soundsStuffed.Length;
        if(index < g_soundsStuffed.Length) {
            CreateTimer(delay, PlaySoundStuffed, index, TIMER_FLAG_NO_MAPCHANGE);
        }

    }
    return Plugin_Continue;
}

public Action EventPointCaptured(Handle event, const char[] name, bool dontBroadcast) {
    g_caps++;
} 

public Action PlaySoundRolled(Handle timer, int index) {
    char file[PLATFORM_MAX_PATH];
    g_soundsRolled.GetString(index, file, sizeof(file));
    LogMessage("Rolled! Playing %s", file);
    EmitSoundToAll(file);
}

public Action PlaySoundStuffed(Handle timer, int index) {
    char file[PLATFORM_MAX_PATH];
    g_soundsStuffed.GetString(index, file, sizeof(file));
    LogMessage("Stuffed! Playing %s", file);
    EmitSoundToAll(file);
}

void LoadConfig() {

    g_soundsRolled.Clear();
    g_soundsStuffed.Clear();

    KeyValues cfg = CreateKeyValues("rolls");

    char configFile[PLATFORM_MAX_PATH];
    BuildPath(Path_SM, configFile, sizeof(configFile), CONFIG_FILE);

    LogMessage("loading config file: %s", configFile);
    if(!FileToKeyValues(cfg, configFile)) {
        SetFailState("unable to load config file!");
        return;
    }

    char name[PLATFORM_MAX_PATH];
    char file[PLATFORM_MAX_PATH];
    char downloadFile[PLATFORM_MAX_PATH];

    if(cfg.JumpToKey("Rolled")) {
        KvGotoFirstSubKey(cfg);
        do {
            cfg.GetSectionName(name, sizeof(name));
            cfg.GetString("file", file, sizeof(file));
            g_soundsRolled.PushString(file);

            Format(downloadFile, sizeof(downloadFile), "sound/%s", file);
            AddFileToDownloadsTable(downloadFile);
            PrecacheSound(file, true);

            LogMessage("loaded rolled %s as file %s", name, file);
        } while(KvGotoNextKey(cfg));
    }

    cfg.Rewind();

    if(cfg.JumpToKey("Stuffed")) {
        KvGotoFirstSubKey(cfg);
        do {
            cfg.GetSectionName(name, sizeof(name));
            cfg.GetString("file", file, sizeof(file));
            g_soundsStuffed.PushString(file);

            Format(downloadFile, sizeof(downloadFile), "sound/%s", file);
            AddFileToDownloadsTable(downloadFile);
            PrecacheSound(file, true);

            LogMessage("loaded stuffed %s as file %s", name, file);
        } while(KvGotoNextKey(cfg));
    }

    cfg.Rewind();

    // per map config bits
    if(cfg.JumpToKey("maps")) {
        float time;
        int cap;
        KvGotoFirstSubKey(cfg);
        do {
            cfg.GetSectionName(name, sizeof(name));

            // I'm not seeing a way to see if a key exists. To 
            // work around this give a bogus default value and
            // if we get that as the value, assume the key 
            // doesn't exist.
            time = cfg.GetFloat("time_rolled", KV_FLOAT_UNSET);
            if(time != KV_FLOAT_UNSET) {
                g_smTimeRolled.SetValue(name, time);
            }

            time = cfg.GetFloat("time_stuffed", KV_FLOAT_UNSET);
            if(time != KV_FLOAT_UNSET) {
                g_smTimeStuffed.SetValue(name, time);
            }

            cap = cfg.GetNum("cap_stuffed", KV_INT_UNSET);
            if(cap != KV_INT_UNSET) {
                g_smCapStuffed.SetValue(name, cap);
            }
        } while(KvGotoNextKey(cfg));
    }

    CloseHandle(cfg);
}