#pragma semicolon 1
#pragma newdecls required

#include <sourcemod>
#include <sdkhooks>
#include <sdktools_trace>
#include <sdktools_functions>
#include <tf2_stocks>

#define PLUGIN_AUTHOR  "ack"
#define PLUGIN_VERSION "0.4"

#define CONFIG_FILE    "configs/eotl_ttm.cfg"

#define PLAYER_WIDTH    49
#define PLAYER_HEIGHT   83

public Plugin myinfo = {
	name = "eotl_ttm",
	author = PLUGIN_AUTHOR,
	description = "To The Moon, play a sound if player is flying high/fast",
	version = PLUGIN_VERSION,
	url = ""
};

enum struct PlayerState {
    float lastTrigger;
    float triggerPercent;
}

PlayerState g_playerStates[MAXPLAYERS + 1];
ConVar g_cvMinSpeed;
ConVar g_cvMinZAngle;
ConVar g_cvMinTime;
ConVar g_cvMinRunway;

ConVar g_cvDemomanPercent;
ConVar g_cvEngineerPercent;
ConVar g_cvHeavyPercent;
ConVar g_cvMedicPercent;
ConVar g_cvPyroPercent;
ConVar g_cvScoutPercent;
ConVar g_cvSniperPercent;
ConVar g_cvSoldierPercent;
ConVar g_cvSpyPercent;

StringMap g_smSounds;
ArrayList g_alSounds;

public void OnPluginStart() {
    LogMessage("version %s starting", PLUGIN_VERSION);

    HookEvent("player_spawn", EventPlayerSpawn);

    g_cvMinSpeed = CreateConVar("eotl_ttm_min_speed", "750.0", "How fast the player must be traveling", 0, true, 1.0);
    g_cvMinZAngle = CreateConVar("eotl_ttm_min_z_angle", "20.0", "Minimum angle above the horizon the player must be traveling", 0, true, 1.0, true, 89.0);
    g_cvMinTime = CreateConVar("eotl_ttm_min_time", "5.0", "Minimum amount time between ttm triggers for a given client", 0, true, 0.0);
    g_cvMinRunway = CreateConVar("eotl_ttm_min_runway", "500.0", "Minimum distance the player must be able to travel at their current location + velocity vector", 0, true, 1.0);

    g_cvDemomanPercent = CreateConVar("eotl_ttm_demoman_percent", "20.0", "Percent chance to trigger ttm", 0, true, 0.0, true, 100.0);
    g_cvEngineerPercent = CreateConVar("eotl_ttm_engineer_percent", "100.0", "Percent chance to trigger ttm", 0, true, 0.0, true, 100.0);
    g_cvHeavyPercent = CreateConVar("eotl_ttm_heavy_percent", "100.0", "Percent chance to trigger ttm", 0, true, 0.0, true, 100.0);
    g_cvMedicPercent = CreateConVar("eotl_ttm_medic_percent", "100.0", "Percent chance to trigger ttm", 0, true, 0.0, true, 100.0);
    g_cvPyroPercent = CreateConVar("eotl_ttm_pyro_percent", "100.0", "Percent chance to trigger ttm", 0, true, 0.0, true, 100.0);
    g_cvScoutPercent = CreateConVar("eotl_ttm_scout_percent", "100.0", "Percent chance to trigger ttm", 0, true, 0.0, true, 100.0);
    g_cvSniperPercent = CreateConVar("eotl_ttm_sniper_percent", "100.0", "Percent chance to trigger ttm", 0, true, 0.0, true, 100.0);
    g_cvSoldierPercent = CreateConVar("eotl_ttm_solider_percent", "20.0", "Percent chance to trigger ttm", 0, true, 0.0, true, 100.0);
    g_cvSpyPercent = CreateConVar("eotl_ttm_spy_percent", "100.0", "Percent chance to trigger ttm", 0, true, 0.0, true, 100.0);

    g_alSounds = CreateArray(PLATFORM_MAX_PATH);
}

public void OnMapStart() {
    g_smSounds = CreateTrie();
    g_alSounds.Clear();

    LoadConfig();
}

public void OnMapEnd() {
    CloseHandle(g_smSounds);
}

public void OnClientPutInServer(int client) {
    SDKHook(client, SDKHook_PostThinkPost, OnClientPostThinkPost);
    g_playerStates[client].lastTrigger = 0.0;
}

public Action EventPlayerSpawn(Handle event, const char[] name, bool dontBroadcast) {
    int client = GetClientOfUserId(GetEventInt(event, "userid"));
	
    g_playerStates[client].triggerPercent = GetTriggerPercent(client);
}

// determine if ttm is disabled for the clients class
float GetTriggerPercent(int client) {

    TFClassType class = TF2_GetPlayerClass(client);

    switch(class) {
        case TFClass_DemoMan:
            return g_cvDemomanPercent.FloatValue;
        case TFClass_Engineer:
            return g_cvEngineerPercent.FloatValue;
        case TFClass_Heavy:
            return g_cvHeavyPercent.FloatValue;
        case TFClass_Medic:
            return g_cvMedicPercent.FloatValue;
        case TFClass_Pyro:
            return g_cvPyroPercent.FloatValue;
        case TFClass_Scout:
            return g_cvScoutPercent.FloatValue;
        case TFClass_Sniper:
            return g_cvSniperPercent.FloatValue;
        case TFClass_Soldier:
            return g_cvSoldierPercent.FloatValue;
        case TFClass_Spy:
            return g_cvSpyPercent.FloatValue;
    }

    LogMessage("client: %d has unknown class %d", client, view_as<int>(class));
    return 0.0;
}

public void OnClientPostThinkPost(int client) {

    float gameTime = GetGameTime();

    if(!IsPlayerAlive(client)) {
        return;
    }

    // hasn't been long enough since the last time this client triggerred
    if(g_playerStates[client].lastTrigger + g_cvMinTime.FloatValue > gameTime) {
        return;
    }

    float velocity[3];
    GetEntPropVector(client, Prop_Data, "m_vecVelocity", velocity);
    // client has no upward velocity
    if(velocity[2] <= 0) {
        return;
    }

    float speed = GetVectorLength(velocity, false);
    if(speed < g_cvMinSpeed.FloatValue) {
       return;
    }

    float zAngle = RadToDeg(ArcSine(velocity[2] / speed));
    if(zAngle < g_cvMinZAngle.FloatValue) {
       return;
    }

    float origin[3];
    GetClientAbsOrigin(client, origin);
    float runway = PlayerRunwayDistance(client, origin, velocity);

    // even if the player doesn't have enough runway distance we should
    // flag them as having triggerred.  The distance is unlikely to go
    // up for the current trajectory so no point in checking over and over.
    g_playerStates[client].lastTrigger = gameTime;

    if(runway <= g_cvMinRunway.FloatValue) {
        return;
    }

    float randomTrigger = GetRandomFloat(0.0, 100.0);
    if(randomTrigger > g_playerStates[client].triggerPercent) {
        return;
    }

    LogMessage("trigger! client: %d, speed: %.1f, z angle: %.1f%, runway: %.1f", client, speed, zAngle, runway);

    int rand = GetURandomInt();
    int index = rand % g_alSounds.Length;

    char soundFile[PLATFORM_MAX_PATH];
    int playType;
    g_alSounds.GetString(index, soundFile, sizeof(soundFile));
    if(!g_smSounds.GetValue(soundFile, playType)) {
        LogMessage("ERROR: unable to look up playType for %s, forcing 0", soundFile);
        playType = 0;
    }

    LogMessage("playing %s with playType: %d", soundFile, playType);
    if(playType == 0) {
        EmitSoundToAll(soundFile, client);
    } else {
        EmitSoundToAll(soundFile, SOUND_FROM_WORLD, _, _, _, _, _, _, origin);
    }
}

// given a players origin and velocity vector return how far they
// could travel before then run into something.
#define BOX_LOWER_LEFT  0
#define BOX_LOWER_RIGHT 1
#define BOX_UPPER_LEFT  2
#define BOX_UPPER_RIGHT 3
float PlayerRunwayDistance(int client, float origin[3], float velocity[3]) {
    // we are creating a 4 point box that roughly represents the
    // players width and height.  This box/plane in the xy axis
    // need to be perpendicular to the xy of the velocity vector.
    float xyVelocity[3];
    float traceBox[4][3];
    CopyVector(velocity, xyVelocity);
    xyVelocity[2] = 0.0;

    CopyVector(origin, traceBox[BOX_LOWER_LEFT]);
    CopyVector(origin, traceBox[BOX_LOWER_RIGHT]);

    NormalizeVector(xyVelocity, xyVelocity);
    ScaleVector(xyVelocity, PLAYER_WIDTH / 2.0);

    traceBox[BOX_LOWER_LEFT][0] = traceBox[BOX_LOWER_LEFT][0] - xyVelocity[0];
    traceBox[BOX_LOWER_LEFT][1] = traceBox[BOX_LOWER_LEFT][1] + xyVelocity[1];
    traceBox[BOX_LOWER_RIGHT][0] = traceBox[BOX_LOWER_RIGHT][0] + xyVelocity[0];
    traceBox[BOX_LOWER_RIGHT][1] = traceBox[BOX_LOWER_RIGHT][1] - xyVelocity[1];

    CopyVector(traceBox[BOX_LOWER_LEFT], traceBox[BOX_UPPER_LEFT]);
    CopyVector(traceBox[BOX_LOWER_RIGHT], traceBox[BOX_UPPER_RIGHT]);

    traceBox[BOX_UPPER_LEFT][2] = traceBox[BOX_UPPER_LEFT][2] + PLAYER_HEIGHT; 
    traceBox[BOX_UPPER_RIGHT][2] = traceBox[BOX_UPPER_RIGHT][2] + PLAYER_HEIGHT; 

    // at this point we have our 4 points of the box, now we need to do a trace
    // from those point using the original velocity vector and return which is
    // the shortest distance.
    float direction[3];
    GetVectorAngles(velocity, direction);
    float runway = -1.0;
    float distance;
    for(int i = 0; i < 4; i++) {
        distance = GetDistance(client, traceBox[i], direction);
        if(runway == -1 || distance < runway) {
            runway = distance;
        }
    }
    return runway;
}

float GetDistance(int client, float origin[3], float direction[3]) {

    TR_TraceRayFilter(origin, direction, MASK_PLAYERSOLID, RayType_Infinite, TR_FilterSelf, client);
    // can this happen? trace should hit something right?
    if(!TR_DidHit()) {
        return -1.0;
    }
    
    float end[3];
    TR_GetEndPosition(end);
    return GetVectorDistance(origin, end);
}

void CopyVector(float src[3], float dst[3]) {
    dst[0] = src[0];
    dst[1] = src[1];
    dst[2] = src[2];
}

// filter ourself from the trace collision
public bool TR_FilterSelf(int entity, int mask, int client) {
    if(entity == client) {
        return false;
    }
    return true;
}

void LoadConfig() {
    KeyValues cfg = CreateKeyValues("ttm");

    char configFile[PLATFORM_MAX_PATH];
    BuildPath(Path_SM, configFile, sizeof(configFile), CONFIG_FILE);

    LogMessage("loading config file: %s", configFile);
    if(!FileToKeyValues(cfg, configFile)) {
        SetFailState("unable to load config file!");
        return;
    }

    char soundFile[PLATFORM_MAX_PATH];
    char downloadFile[PLATFORM_MAX_PATH];
    int playType;
    KvGotoFirstSubKey(cfg);
    do {
        cfg.GetString("soundFile", soundFile, sizeof(soundFile));
        playType = cfg.GetNum("playType", 0);

        if(playType != 0 && playType != 1) {
            LogError("WARN: Invalid playType %d on %s in config, forcing 0", playType, soundFile);
            playType = 0;
        }

        if(!SetTrieValue(g_smSounds, soundFile, playType)) {
            LogError("WARN: %s soundFile is defined twice in the config", soundFile);
            continue;
        }
        g_alSounds.PushString(soundFile);

        Format(downloadFile, sizeof(downloadFile), "sound/%s", soundFile);
        AddFileToDownloadsTable(downloadFile);
        PrecacheSound(soundFile, true);
        LogMessage("soundFile: %s, playType: %d", soundFile, playType);
    } while(KvGotoNextKey(cfg));

    CloseHandle(cfg);  
}